--[[
    GF_Auction UISkin Module
    Detects and applies UI skin based on ElvUI, NDui, or default Blizzard style
]]

local UISkin = {}

if not GF_Auction then
    GF_Auction = {}
end
GF_Auction.UISkin = UISkin

local currentStyle = "BLIZZARD"

local function DetectStyle()
    local success, result = pcall(function()
        if _G.ElvUI and type(_G.ElvUI) == "table" and _G.ElvUI[1] then
            return "ELVUI"
        elseif IsAddOnLoaded("ElvUI") and _G.ElvUI then
            return "ELVUI"
        elseif _G.NDui and type(_G.NDui) == "table" and _G.NDui[1] then
            return "NDUI"
        elseif IsAddOnLoaded("NDui") and _G.NDui then
            return "NDUI"
        else
            return "BLIZZARD"
        end
    end)
    
    if success then
        return result
    else
        return "BLIZZARD"
    end
end

function UISkin:Initialize()
    local success, err = pcall(function()
        currentStyle = DetectStyle()
    end)
    
    if not success then
        currentStyle = "BLIZZARD"
    end
    
    return true
end

function UISkin:RefreshStyle()
    currentStyle = DetectStyle()
end

function UISkin:GetStyle()
    return currentStyle
end

function UISkin:IsElvUI()
    return currentStyle == "ELVUI"
end

function UISkin:IsNDui()
    return currentStyle == "NDUI"
end

function UISkin:IsBlizzard()
    return currentStyle == "BLIZZARD"
end

function UISkin:CreateFrame(frameType, name, parent, template)
    local style = self:GetStyle()
    
    if style == "ELVUI" then
        local S, E = self:GetElvUISkin()
        if S then
            local frame = CreateFrame(frameType, name, parent, template)
            if frame and S.HandleFrame then
                local success = pcall(function()
                    S:HandleFrame(frame, true)
                end)
                if not success then
                    -- Fallback to default if ElvUI skin fails
                end
            end
            return frame
        end
    elseif style == "NDUI" and _G.NDui then
        local frame = CreateFrame(frameType, name, parent, template)
        if frame and NDuiDB and NDuiDB["Skins"] then
            if NDuiDB["Skins"]["Enable"] then
                if NDuiDB["Skins"]["FlatMode"] then
                    -- Apply flat mode skin
                else
                    -- Apply normal NDui skin
                end
            end
        end
        return frame
    end
    
    return CreateFrame(frameType, name, parent, template)
end

function UISkin:GetBackdrop()
    local style = self:GetStyle()
    
    if style == "ELVUI" and _G.ElvUI then
        local E = _G.ElvUI[1]
        if E and E.media and E.media.blank and E.mult then
            return {
                bgFile = E.media.blank,
                edgeFile = E.media.blank,
                tile = false,
                tileSize = 0,
                edgeSize = E.mult,
                insets = {
                    left = -E.mult,
                    right = -E.mult,
                    top = -E.mult,
                    bottom = -E.mult
                }
            }
        end
    elseif style == "NDUI" and _G.NDui then
        return {
            bgFile = "Interface\\Buttons\\WHITE8X8",
            edgeFile = "Interface\\Buttons\\WHITE8X8",
            tile = false,
            tileSize = 0,
            edgeSize = 1,
            insets = { left = 0, right = 0, top = 0, bottom = 0 }
        }
    end
    
    return {
        bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background",
        edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Border",
        tile = true,
        tileSize = 32,
        edgeSize = 32,
        insets = { left = 11, right = 12, top = 12, bottom = 11 }
    }
end

function UISkin:GetBackdropColor()
    local style = self:GetStyle()
    
    if style == "ELVUI" and _G.ElvUI then
        local E = _G.ElvUI[1]
        if E and E.media and E.media.backdropcolor then
            local r, g, b = unpack(E.media.backdropcolor)
            local fade = (E.private and E.private.general and E.private.general.backdropfade) and 0.8 or 1
            return r, g, b, fade
        end
    elseif style == "NDUI" then
        return 0, 0, 0, 0.8
    end
    
    return 0, 0, 0, 0.92
end

function UISkin:GetBackdropBorderColor()
    local style = self:GetStyle()
    
    if style == "ELVUI" and _G.ElvUI then
        local E = _G.ElvUI[1]
        if E and E.private and E.private.general then
            return unpack(E.media.bordercolor)
        end
    elseif style == "NDUI" then
        return 0.3, 0.3, 0.3, 1
    end
    
    return 0.5, 0.5, 0.5, 1
end

function UISkin:GetTitleColor()
    local style = self:GetStyle()
    
    if style == "ELVUI" and _G.ElvUI then
        local E = _G.ElvUI[1]
        if E and E.private and E.private.general then
            return unpack(E.media.rgbvaluecolor)
        end
    elseif style == "NDUI" then
        return 1, 0.82, 0, 1
    end
    
    return 1.0, 0.82, 0.0, 1
end

function UISkin:GetTabStyle()
    local style = self:GetStyle()
    
    if style == "ELVUI" then
        return {
            selectedColor = {1, 0.82, 0, 1},
            unselectedColor = {0.6, 0.6, 0.6, 1},
            height = 25
        }
    elseif style == "NDUI" then
        return {
            selectedColor = {1, 0.82, 0, 1},
            unselectedColor = {0.6, 0.6, 0.6, 1},
            height = 25
        }
    end
    
    return {
        selectedColor = {1, 1, 1, 1},
        unselectedColor = {0.8, 0.8, 0.8, 1},
        height = 25
    }
end

function UISkin:ApplyBackdrop(frame)
    if not frame or not frame.SetBackdrop then
        return
    end
    
    local backdrop = self:GetBackdrop()
    frame:SetBackdrop(backdrop)
    
    local r, g, b, a = self:GetBackdropColor()
    frame:SetBackdropColor(r, g, b, a)
    
    local br, bg, bb, ba = self:GetBackdropBorderColor()
    frame:SetBackdropBorderColor(br, bg, bb, ba)
end

function UISkin:GetElvUISkin()
    if not self:IsElvUI() or not _G.ElvUI then
        return nil, nil
    end
    
    local E = _G.ElvUI[1]
    if not E then
        return nil, nil
    end
    
    local success, S = pcall(function()
        return E:GetModule('Skins')
    end)
    
    if not success or not S then
        return nil, E
    end
    
    return S, E
end

function UISkin:HandleCloseButton(button, point, x, y)
    if not button then
        return
    end
    
    local style = self:GetStyle()
    
    if style == "ELVUI" then
        local S = self:GetElvUISkin()
        if S and S.HandleCloseButton then
            local success = pcall(function()
                S:HandleCloseButton(button, point, x, y)
            end)
            if success then
                return
            end
        end
    elseif style == "NDUI" and _G.NDui then
        local B = _G.NDui[1]
        if B and B.ReskinClose then
            local success = pcall(function()
                B:ReskinClose(button, point, x, y)
            end)
            if success then
                return
            end
        end
    end
end

function UISkin:HandleButton(button)
    if not button then
        return
    end
    
    local style = self:GetStyle()
    
    if style == "ELVUI" then
        local S = self:GetElvUISkin()
        if S and S.HandleButton then
            local success = pcall(function()
                S:HandleButton(button)
            end)
            if success then
                return
            end
        end
    elseif style == "NDUI" and _G.NDui then
        local B = _G.NDui[1]
        if B and B.Reskin then
            local success = pcall(function()
                B:Reskin(button)
            end)
            if success then
                return
            end
        end
    end
end

function UISkin:HandleScrollBar(scrollBar)
    if not scrollBar then
        return
    end
    
    local style = self:GetStyle()
    
    if style == "ELVUI" then
        local S = self:GetElvUISkin()
        if S and S.HandleScrollBar then
            local success = pcall(function()
                S:HandleScrollBar(scrollBar)
            end)
            if success then
                return
            end
        end
    elseif style == "NDUI" and _G.NDui then
        local B = _G.NDui[1]
        if B and B.ReskinScroll then
            local success = pcall(function()
                B:ReskinScroll(scrollBar)
            end)
            if success then
                return
            end
        end
    end
end

function UISkin:HandleScrollFrame(scrollFrame)
    if not scrollFrame then
        return
    end
    
    local style = self:GetStyle()
    
    if style == "ELVUI" then
        local S = self:GetElvUISkin()
        if S then
            if scrollFrame.ScrollBar then
                local success = pcall(function()
                    S:HandleScrollBar(scrollFrame.ScrollBar)
                end)
                if success then
                    return
                end
            end
        end
    elseif style == "NDUI" and _G.NDui then
        local B = _G.NDui[1]
        if B and scrollFrame.ScrollBar then
            if B.ReskinScroll then
                local success = pcall(function()
                    B:ReskinScroll(scrollFrame.ScrollBar)
                end)
                if success then
                    return
                end
            end
        end
    end
end

function UISkin:HandleFrame(frame, setBackdrop)
    if not frame then
        return
    end
    
    local style = self:GetStyle()
    
    if style == "ELVUI" then
        local S = self:GetElvUISkin()
        if S and S.HandleFrame then
            local success = pcall(function()
                S:HandleFrame(frame, setBackdrop or true)
            end)
            if success then
                return true
            end
        end
    elseif style == "NDUI" and _G.NDui then
        local B = _G.NDui[1]
        if B and B.ReskinPortraitFrame then
            local success = pcall(function()
                B:ReskinPortraitFrame(frame)
            end)
            if success then
                return true
            end
        elseif B and B.SetBD then
            local success = pcall(function()
                B:SetBD(frame)
            end)
            if success then
                return true
            end
        end
    end
    
    return false
end

function UISkin:HandleTab(tab)
    if not tab then
        return
    end
    
    local style = self:GetStyle()
    
    if style == "ELVUI" then
        local S = self:GetElvUISkin()
        if S and S.HandleTab then
            local success = pcall(function()
                S:HandleTab(tab)
            end)
            if success then
                return
            end
        end
    elseif style == "NDUI" and _G.NDui then
        local B = _G.NDui[1]
        if B and B.ReskinTab then
            local success = pcall(function()
                B:ReskinTab(tab)
            end)
            if success then
                return
            end
        end
    end
end

