-- Based on AceGUIWidget-EditBox supplied with AceGUI-3.0

local Type, Version = "LSPreviewBox", 1
local AceGUI = LibStub("AceGUI-3.0")
if not AceGUI or (AceGUI:GetWidgetVersion(Type) or 0) >= Version then return end

-- Lua
local _G = getfenv(0)
local next = _G.next
local tonumber = _G.tonumber
local tostring = _G.tostring

-- Blizz
local C_CurrencyInfo = _G.C_CurrencyInfo

--[[ luacheck: globals
ChatFontNormal ClearCursor CreateFrame GetCursorInfo GetMacroInfo GetSpellInfo PlaySound UIParent

OKAY
]]

local function ShowButton(self)
	if not self.disablebutton then
		self.button:Show()
		self.editbox:SetTextInsets(0, 20, 3, 3)
	end
end

local function HideButton(self)
	self.button:Hide()
	self.editbox:SetTextInsets(0, 0, 3, 3)
end

local function Control_OnEnter(frame)
	frame.obj:Fire("OnEnter")
end

local function Control_OnLeave(frame)
	frame.obj:Fire("OnLeave")
end

local function Frame_OnShowFocus(frame)
	frame.obj.editbox:SetFocus()
	frame:SetScript("OnShow", nil)
end

local function EditBox_OnEscapePressed(frame)
	AceGUI:ClearFocus()
end

local function EditBox_OnEnterPressed(frame)
	local self = frame.obj
	local value = frame:GetText()
	local cancel = self:Fire("OnEnterPressed", value)
	if not cancel then
		PlaySound(856) -- SOUNDKIT.IG_MAINMENU_OPTION_CHECKBOX_ON
		HideButton(self)
	end
end

local function EditBox_OnReceiveDrag(frame)
	local self = frame.obj
	local type, id, info = GetCursorInfo()
	local name
	if type == "item" then
		name = info
	elseif type == "spell" then
		name = GetSpellInfo(id, info)
	elseif type == "macro" then
		name = GetMacroInfo(id)
	end
	if name then
		self:SetText(name)
		self:Fire("OnEnterPressed", name)
		ClearCursor()
		HideButton(self)
		AceGUI:ClearFocus()
	end
end

local function EditBox_OnTextChanged(frame)
	local self = frame.obj
	local value = frame:GetText()
	if tostring(value) ~= tostring(self.lasttext) then
		self:Fire("OnTextChanged", value)
		self:SetPreview(value)
		self.lasttext = value
		ShowButton(self)
	end
end

local function EditBox_OnFocusGained(frame)
	AceGUI:SetFocus(frame.obj)
end

local function Button_OnClick(frame)
	local editbox = frame.obj.editbox
	editbox:ClearFocus()
	EditBox_OnEnterPressed(editbox)
end

local methods = {
	["OnAcquire"] = function(self)
		-- height is controlled by SetLabel
		self:SetWidth(200)
		self:SetDisabled(false)
		self:SetLabel()
		self:SetText()
		self:SetPreview()
		self:DisableButton(false)
		self:SetMaxLetters(0)
	end,

	["OnRelease"] = function(self)
		self:SetText()
		self:ClearFocus()
	end,

	["SetDisabled"] = function(self, disabled)
		self.disabled = disabled
		if disabled then
			self.editbox:EnableMouse(false)
			self.editbox:ClearFocus()
			self.editbox:SetTextColor(0.5,0.5,0.5)
			self.label:SetTextColor(0.5,0.5,0.5)
		else
			self.editbox:EnableMouse(true)
			self.editbox:SetTextColor(1,1,1)
			self.label:SetTextColor(1,.82,0)
		end
	end,

	["SetText"] = function(self, text)
		self.lasttext = text or ""
		self.editbox:SetText(text or "")
		self.editbox:SetCursorPosition(0)
		HideButton(self)
	end,

	["GetText"] = function(self)
		return self.editbox:GetText()
	end,

	["SetLabel"] = function(self, text)
		if text and text ~= "" then
			self.label:SetText(text)
			self.label:Show()
			self.editbox:SetPoint("TOPLEFT",self.frame,"TOPLEFT",7,-18)
			self:SetHeight(44)
			self.alignoffset = 30
		else
			self.label:SetText("")
			self.label:Hide()
			self.editbox:SetPoint("TOPLEFT",self.frame,"TOPLEFT",7,0)
			self:SetHeight(26)
			self.alignoffset = 12
		end
	end,

	["DisableButton"] = function(self, disabled)
		self.disablebutton = disabled
		if disabled then
			HideButton(self)
		end
	end,

	["SetMaxLetters"] = function (self, num)
		self.editbox:SetMaxLetters(num or 0)
	end,

	["ClearFocus"] = function(self)
		self.editbox:ClearFocus()
		self.frame:SetScript("OnShow", nil)
	end,

	["SetFocus"] = function(self)
		self.editbox:SetFocus()
		if not self.frame:IsShown() then
			self.frame:SetScript("OnShow", Frame_OnShowFocus)
		end
	end,

	["HighlightText"] = function(self, from, to)
		self.editbox:HighlightText(from, to)
	end,

	["SetPreview"] = function(self, value)
		if value and value ~= "" then
			self.preview:SetOwner(self, "ANCHOR_NONE")
			self.preview:SetPoint("TOPLEFT", self, "BOTTOMLEFT", 1, 1)
			self.preview:SetText(value, 1, 0.82, 0, true)
			self.preview:Show()
		else
			self.preview:Hide()
		end
	end,
}

local function getBaseWidget(widgetType)
	local frame = CreateFrame("Frame", nil, UIParent)
	frame:Hide()

	local name = "AceGUI30" .. widgetType .. AceGUI:GetNextWidgetNum(Type)

	local editbox = CreateFrame("EditBox", name, frame, "InputBoxTemplate")
	editbox:SetAutoFocus(false)
	editbox:SetFontObject(ChatFontNormal)
	editbox:SetScript("OnEnter", Control_OnEnter)
	editbox:SetScript("OnLeave", Control_OnLeave)
	editbox:SetScript("OnEscapePressed", EditBox_OnEscapePressed)
	editbox:SetScript("OnEnterPressed", EditBox_OnEnterPressed)
	editbox:SetScript("OnTextChanged", EditBox_OnTextChanged)
	editbox:SetScript("OnReceiveDrag", EditBox_OnReceiveDrag)
	editbox:SetScript("OnMouseDown", EditBox_OnReceiveDrag)
	editbox:SetScript("OnEditFocusGained", EditBox_OnFocusGained)
	editbox:SetTextInsets(0, 0, 3, 3)
	editbox:SetMaxLetters(256)
	editbox:SetPoint("BOTTOMLEFT", 6, 0)
	editbox:SetPoint("BOTTOMRIGHT")
	editbox:SetHeight(19)

	local label = frame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
	label:SetPoint("TOPLEFT", 0, -2)
	label:SetPoint("TOPRIGHT", 0, -2)
	label:SetJustifyH("LEFT")
	label:SetHeight(18)

	local preview = CreateFrame("GameTooltip", name .. "Preview", frame, "GameTooltipTemplate")
	preview:SetFrameLevel(99) -- it should be above other widgets
	preview:Hide()

	local button = CreateFrame("Button", nil, editbox, "UIPanelButtonTemplate")
	button:SetWidth(40)
	button:SetHeight(20)
	button:SetPoint("RIGHT", -2, 0)
	button:SetText(OKAY)
	button:SetScript("OnClick", Button_OnClick)
	button:Hide()

	return {
		alignoffset = 30,
		editbox     = editbox,
		label       = label,
		preview     = preview,
		button      = button,
		frame       = frame,
		type        = widgetType
	}
end

--------------
-- Subtypes --
--------------
do
	local Subtype = Type .. "Currency"

	local function previewCurrency(self, value)
		local id = tonumber(value)
		if id then
			id = "currency:" .. id
			local info = C_CurrencyInfo.GetCurrencyInfoFromLink(id, 0)
			if info then
				self.preview:SetOwner(self.frame, "ANCHOR_NONE")
				self.preview:SetPoint("TOPLEFT", self.frame, "BOTTOMLEFT", 1, 1)
				self.preview:SetHyperlink(id)
				self.preview:Show()
			else
				self.preview:Hide()
			end
		else
			self.preview:Hide()
		end
	end

	local function Constructor()
		local widget = getBaseWidget(Subtype)

		for method, func in next, methods do
			widget[method] = func
		end

		widget.SetPreview = previewCurrency

		widget.editbox.obj = widget
		widget.button.obj = widget

		return AceGUI:RegisterAsWidget(widget)
	end

	AceGUI:RegisterWidgetType(Subtype, Constructor, Version)
end

do
	local Subtype = Type .. "Item"

	local function previewItem(self, value)
		local id = tonumber(value)
		if id then
			id = "item:" .. id
			local info = GetItemInfoInstant(id)
			if info then
				self.preview:SetOwner(self.frame, "ANCHOR_NONE")
				self.preview:SetPoint("TOPLEFT", self.frame, "BOTTOMLEFT", 1, 1)
				self.preview:SetHyperlink(id)
				self.preview:Show()
			else
				self.preview:Hide()
			end
		else
			self.preview:Hide()
		end
	end

	local function Constructor()
		local widget = getBaseWidget(Subtype)

		for method, func in next, methods do
			widget[method] = func
		end

		widget.SetPreview = previewItem

		widget.editbox.obj = widget
		widget.button.obj = widget

		return AceGUI:RegisterAsWidget(widget)
	end

	AceGUI:RegisterWidgetType(Subtype, Constructor, Version)
end
